// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <biquad.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

// The calculations here are fixed point with 28 bits after the point.
#define FBITS 28

#define FLOAT_TO_FIXED32(x, y) (int32_t) (((float) x) * ((float) (1 << y)))
#define DOUBLE_TO_FIXED32(x, y) (int32_t) (((double) x) * ((double) (1 << y)))
#define FIXED32_TO_FLOAT(x, y) (((float) x) / ((float) (1<<y)))
#define FIXED32_TO_DOUBLE(x, y) (((double) x) / ((double) (1<<y)))

/* Compute sin in the range -2pi .. 2pi in 4.28 fixed point */
static int32_t bq_sin(int32_t x)
{
    static const
    int32_t a1 = DOUBLE_TO_FIXED32(-0.1666665668, FBITS),
            a2 = DOUBLE_TO_FIXED32( 0.8333025139e-02, FBITS),
            a3 = DOUBLE_TO_FIXED32(-0.1980741872e-03, FBITS),
            a4 = DOUBLE_TO_FIXED32(0.2601903036e-5, FBITS);
    const int32_t pi = DOUBLE_TO_FIXED32(M_PI, FBITS);
    int sign = 1;
    // Reduce the range
    if (x < 0) {
      x = -x;
      sign = -1;
    }
    if (x > pi) {
      x = x - pi;
      sign = -sign;
    }
    if (x > pi/2) {
      x = pi - x;
    }
    int64_t x2 = ((int64_t) x * x) >> FBITS;
    int64_t acc;
    acc = (x2 * a4) >> FBITS;
    acc = (x2 * (a3 + acc)) >> FBITS;
    acc = (x2 * (a2 + acc)) >> FBITS;
    acc = (x2 * (a1 + acc)) >> FBITS;
    acc = (((int64_t) x << FBITS) + x * acc) >> FBITS;
    return (int32_t) sign * acc;
}

/* Compute cos in the range -2pi .. 2pi in 4.28 fixed point */
static int32_t bq_cos(int32_t x) {
  const int32_t pi = DOUBLE_TO_FIXED32(M_PI, FBITS);
  x = x + pi/2;
  if (x > 2*pi)
    x -= 2*pi;
  return bq_sin(x);
}


int32_t apply_biquad(int32_t val, biquad_state_t &st)
{
  int64_t res;

  res = ((int64_t) st.b[0]) * val +
        ((int64_t) st.b[1]) * st.prev_inputs[0] +
        ((int64_t) st.b[2]) * st.prev_inputs[1] -
        ((int64_t) st.a[0]) * st.prev_outputs[0] -
        ((int64_t) st.a[1]) * st.prev_outputs[1] +
        st.error;

  int64_t scaled_res = res >> FBITS;

  if (scaled_res > INT32_MAX)
    scaled_res = INT32_MAX;
  if (scaled_res < INT32_MIN)
    scaled_res = INT32_MIN;

  st.error = res - (scaled_res << FBITS);
  st.prev_inputs[1] = st.prev_inputs[0];
  st.prev_inputs[0] = val;
  st.prev_outputs[1] = st.prev_outputs[0];
  st.prev_outputs[0] = (int32_t) scaled_res;

  return (int32_t) scaled_res;
}

void init_biquad_state(enum biquad_type_t type,
                       unsigned significant_freq,
                       unsigned sample_freq,
                       uint32_t Q,
                       biquad_state_t &st)
{
  int32_t w0 =
    (2 * DOUBLE_TO_FIXED32(M_PI, FBITS) * ((int64_t) significant_freq)) / sample_freq;
  int32_t sin_w0 = bq_sin(w0);
  int32_t cos_w0 = bq_cos(w0);
  int32_t alpha = ((int64_t) sin_w0 << BIQUAD_PARAM_FRACTIONAL_BITS) / (2*Q);
  int32_t b0, b1, b2, a0, a1, a2;
  int32_t one = (1<<FBITS), two = (2<<FBITS);

  switch (type) {
  case BIQUAD_LOPASS:
    b0 = (one - cos_w0)/2;
    b1 = (one - cos_w0);
    b2 = (one - cos_w0)/2;
    a0 = one + alpha;
    a1 = (((int64_t) -two) * cos_w0) >> FBITS;
    a2 = (one - alpha);
    break;
  case BIQUAD_HIPASS:
    b0 = (one + cos_w0)/2;
    b1 = -(one + cos_w0);
    b2 = (one + cos_w0)/2;
    a0 = one + alpha;
    a1 = (((int64_t) -two) * cos_w0) >> FBITS;
    a2 = (one - alpha);
    break;
  case BIQUAD_BANDPASS:
    if (Q == 0) {
      b0 = alpha;
      b1 = 0;
      b2 = -alpha;
      a0 = one + alpha;
      a1 = (((int64_t) -two) * cos_w0) >> FBITS;
      a2 = (one - alpha);
    } else {
      b0 = (sin_w0)/2;
      b1 = 0;
      b2 = (-sin_w0)/2;
      a0 = one + alpha;
      a1 = (((int64_t) -two) * cos_w0) >> FBITS;
      a2 = (one - alpha);
    }
    break;
  case BIQUAD_NOTCH:
    b0 = one;
    b1 = (((int64_t) -two) * cos_w0) >> FBITS;
    b2 = one;
    a0 = one + alpha;
    a1 = (((int64_t) -two) * cos_w0) >> FBITS;
    a2 = (one - alpha);
    break;
  case BIQUAD_ALLPASS:
    b0 = one - alpha;
    b1 = (((int64_t) -two) * cos_w0) >> FBITS;
    b2 = one + alpha;
    a0 = one + alpha;
    a1 = (((int64_t) -two) * cos_w0) >> FBITS;
    a2 = one - alpha;
    break;
  }

  b0 = ((int64_t) b0 << FBITS)/a0;
  b1 = ((int64_t) b1 << FBITS)/a0;
  b2 = ((int64_t) b2 << FBITS)/a0;
  a1 = ((int64_t) a1 << FBITS)/a0;
  a2 = ((int64_t) a2 << FBITS)/a0;

#if DEBUG_COEFFICIENTS
  printf("w0:%f, sin(w0):%f, cos(w0):%f, alpha:%f\n",
         FIXED32_TO_FLOAT(w0, FBITS),
         FIXED32_TO_FLOAT(sin_w0, FBITS),
         FIXED32_TO_FLOAT(cos_w0, FBITS),
         FIXED32_TO_FLOAT(alpha, FBITS));
  printf("b0:%f, b1:%f, b2:%f, a1:%f, a2:%f\n",
         FIXED32_TO_FLOAT(b0, FBITS),
         FIXED32_TO_FLOAT(b1, FBITS),
         FIXED32_TO_FLOAT(b2, FBITS),
         FIXED32_TOlFLOAT(a1, FBITS),
         FIXED32_TO_FLOAT(a2, FBITS));
  printf("b0:%d, b1:%d, b2:%d, a1:%d, a2:%d\n", b0, b1, b2, a1, a2);;
#endif

  st.b[0] = b0;
  st.b[1] = b1;
  st.b[2] = b2;
  st.a[0] = a1;
  st.a[1] = a2;
  st.prev_inputs[0] = 0;
  st.prev_inputs[1] = 0;
  st.prev_outputs[0] = 0;
  st.prev_outputs[1] = 0;
  st.error = 0;
}
