// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <xs1.h>
#include "i2c.h"
#include "usb.h"

extern unsigned char g_reportBuffer[4];

// FXOS8700EQ register address defines
#define FXOS8700EQ_I2C_ADDR 0x1E
#define FXOS8700EQ_XYZ_DATA_CFG_REG 0x0E
#define FXOS8700EQ_CTRL_REG_1 0x2A
#define FXOS8700EQ_DR_STATUS 0x0
#define FXOS8700EQ_OUT_X_MSB 0x1
#define FXOS8700EQ_OUT_X_LSB 0x2
#define FXOS8700EQ_OUT_Y_MSB 0x3
#define FXOS8700EQ_OUT_Y_LSB 0x4
#define FXOS8700EQ_OUT_Z_MSB 0x5
#define FXOS8700EQ_OUT_Z_LSB 0x6

int read_acceleration(client interface i2c_master_if i2c, int reg) {
    i2c_regop_res_t result;
    int accel_val = 0;
    unsigned char data = 0;

    // Read MSB data
    data = i2c.read_reg(FXOS8700EQ_I2C_ADDR, reg, result);
    if (result != I2C_REGOP_SUCCESS) {
      return 0;
    }

    accel_val = data << 2;

    // Read LSB data
    data = i2c.read_reg(FXOS8700EQ_I2C_ADDR, reg+1, result);
    if (result != I2C_REGOP_SUCCESS) {
      return 0;
    }

    accel_val |= (data >> 6);

    if (accel_val & 0x200) {
      accel_val -= 1023;
    }

    return accel_val;
}

/*
 * This function responds to the HID requests
 */
void hid_mouse_extended(chanend chan_ep_hid, client interface i2c_master_if i2c)
{
    i2c_regop_res_t result;
    char status_data = 0;

    // Configure FXOS8700EQ
    result = i2c.write_reg(FXOS8700EQ_I2C_ADDR, FXOS8700EQ_XYZ_DATA_CFG_REG, 0x01);
    if (result != I2C_REGOP_SUCCESS) {
      return;
    }

    // Enable FXOS8700EQ
    result = i2c.write_reg(FXOS8700EQ_I2C_ADDR, FXOS8700EQ_CTRL_REG_1, 0x01);
    if (result != I2C_REGOP_SUCCESS) {
      return;
    }

    XUD_ep ep_hid = XUD_InitEp(chan_ep_hid, XUD_EPTYPE_INT);

    while (1)
    {
      /* Unsafe region so we can use shared memory. */
      unsafe {
        char * unsafe p_reportBuffer = g_reportBuffer;
        int x, y;

        // Initialise the HID
        p_reportBuffer[1] = 0;
        p_reportBuffer[2] = 0;

        // Wait for valid accelerometer data
        do {
          status_data = i2c.read_reg(FXOS8700EQ_I2C_ADDR, FXOS8700EQ_DR_STATUS, result);
        } while (!status_data & 0x08);

        // Read x and y axis values
        x = read_acceleration(i2c, FXOS8700EQ_OUT_X_MSB);
        y = read_acceleration(i2c, FXOS8700EQ_OUT_Y_MSB);

        // Set up HID x axis value report
        if (x > 50) {
          p_reportBuffer[1] = 10;
        } else if (x < -50) {
          p_reportBuffer[1] = -10;
        }

        // Set up HID y axis value report
        if (y > 50) {
          p_reportBuffer[2] = -10;
        } else if (y < -50) {
          p_reportBuffer[2] = 10;
        }

        /* Send the buffer off to the host.  Note this will return when complete */
        XUD_SetBuffer(ep_hid, (char *) p_reportBuffer, 4);
      }
    }
}


