// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <xs1.h>
#include <platform.h>
#include <quadflashlib.h>
#include <timer.h>

/*
 * the LEDs on explorerKIT are:
 *
 * Single    : 0x01
 * RGB Blue  : 0x02
 * RGB Green : 0x04
 * RGB Red   : 0x08
 *
 * The 12 bytes stored in the data partition
 * drive various combinations of the 4 LED's'.
 * 0x01 - Single
 * 0x02 - RGB Blue
 * 0x04 - RGB Green
 * 0x08 - RGB Red
 * 0x03 - Single | RGB Blue
 * 0x05 - Single | RGB Green
 * 0x09 - Single | RGB Red
 * 0x06 - RGB Blue | RGB Green
 * 0x0A - RGB Blue | RGB Red
 * 0x0C - RGB Green | RGB Red
 * 0x0F - Single | RGB Blue | RGB Green | RGB Red
 * 0x00 - Off
 */

// Ports for QuadSPI access on explorerKIT.
fl_QSPIPorts ports = {
  PORT_SQI_CS,
  PORT_SQI_SCLK,
  PORT_SQI_SIO,
  on tile[0]: XS1_CLKBLK_1
};

// Port where the leds reside on explorerKIT.
on tile[0] : port led_port = XS1_PORT_4F;

// List of QuadSPI devices that are supported by default.
fl_QuadDeviceSpec deviceSpecs[] =
{
  FL_QUADDEVICE_SPANSION_S25FL116K,
  FL_QUADDEVICE_SPANSION_S25FL132K,
  FL_QUADDEVICE_SPANSION_S25FL164K,
  FL_QUADDEVICE_ISSI_IS25LQ080B,
  FL_QUADDEVICE_ISSI_IS25LQ016B,
  FL_QUADDEVICE_ISSI_IS25LQ032B,
};

// There are 12 bytes stored in the data partition.
#define NUM_LED_BYTES 12

int led_driver (void)
{
  char led_byte_buffer[NUM_LED_BYTES];  // Buffer to hold the 12 bytes read from the data partition.
  int delay = 50;                       // Initial delay 50 ms.
  int led_counter = 0;                  // A counter to count through the led patterns.

  // Connect to the QuadSPI device using the quadflash library function fl_connectToDevice.
  if(fl_connectToDevice(ports, deviceSpecs, sizeof(deviceSpecs)/sizeof(fl_QuadDeviceSpec)) != 0)
  {
    return 1;
  }

  // Read all 12 bytes from offset 0 in the data partition and store them in
  // the buffer led_byte_buffer using the flash library function fl_readData.
  if(fl_readData(0, NUM_LED_BYTES, led_byte_buffer) != 0)
  {
    return 1;
  }

  // Disconnect from the QuadSPI device.
  fl_disconnect();

  while(1)
  {
    delay_milliseconds(delay);                          // Wait.
    delay += 1;                                         // Increase the delay so the LED pattern gets slower.
    led_port <: led_byte_buffer[led_counter];           // Drive the next led pattern.
    led_counter++;                                      // Pick the next pattern.
    if (led_counter == NUM_LED_BYTES)                   // If we are at the last pattern,
    {
      led_counter = 0;                                  // then wrap around.
    }
  }

  return 0;
}

int main(void)
{
  par
  {
    on tile[0]:
    {
      led_driver();
    }

    on tile[1]:
    {
      while(1);
    }
  }

  return 0;
}

