// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include "usb.h"
#include "u_series_adc.h"
#include "hid_mouse_extended.h"

/* Global report buffer, global since used by Endpoint0 core */
extern unsigned char g_reportBuffer[4];

/* Port for ADC triggering */
on USB_TILE: out port p_adc_trig = PORT_ADC_TRIGGER;

#define BITS 5          // Overall precision
#define DEAD_ZONE 2     // Ensure that the mouse is stable when the joystick is not used
#define SENSITIVITY 0   // Sensitivity range 0 - 9
#define SHIFT  (32 - BITS)
#define MASK   ((1 << BITS) - 1)
#define OFFSET (1 << (BITS - 1))

/*
 * This function responds to the HID requests - it moves the pointers x axis based on ADC input
 */
void hid_mouse_extended(chanend c_ep_hid, chanend c_adc)
{
    int initialDone = 0;
    int initialX = 0;
    int initialY = 0;

    /* Initialise the XUD endpoint */
    XUD_ep ep_hid = XUD_InitEp(c_ep_hid, XUD_EPTYPE_BUL);

    /* Configure and enable the ADC in the U device */
    adc_config_t adc_config = { { 0, 0, 0, 0, 0, 0, 0, 0 }, 0, 0, 0 };

    adc_config.input_enable[2] = 1;
    adc_config.input_enable[3] = 1;
    adc_config.samples_per_packet = 2;
    adc_config.bits_per_sample = ADC_32_BPS;
    adc_config.calibration_mode = 0;

    adc_enable(usb_tile, c_adc, p_adc_trig, adc_config);

    /* Unsafe region so we can use shared memory. */
    while (1)
    {
      unsafe {
        char * unsafe p_reportBuffer = g_reportBuffer;
        unsigned data[2];
        int x;
        int y;

        /* Initialise the HID report buffer */
        p_reportBuffer[1] = 0;
        p_reportBuffer[2] = 0;

        /* Get ADC input */
        adc_trigger_packet(p_adc_trig, adc_config);
        adc_read_packet(c_adc, adc_config, data);
        x = data[0];
        y = data[1];

        /* Move horizontal axis of pointer based on ADC val (absolute) */
        x = ((x >> SHIFT) & MASK) - OFFSET - initialX;
        if (x > DEAD_ZONE)
            p_reportBuffer[1] = (x - DEAD_ZONE)/(10 - SENSITIVITY);
        else if (x < -DEAD_ZONE)
            p_reportBuffer[1] = (x + DEAD_ZONE)/(10 - SENSITIVITY);

        /* Move vertical axis of pointer based on ADC val (absolute) */
        y = ((y >> SHIFT) & MASK) - OFFSET - initialY;
        if (y > DEAD_ZONE)
            p_reportBuffer[2] = (y - DEAD_ZONE)/(10 - SENSITIVITY);
        else if (y < -DEAD_ZONE)
            p_reportBuffer[2] = (y + DEAD_ZONE)/(10 - SENSITIVITY);

        if (!initialDone)
        {
            initialX = x;
            initialY = y;
            initialDone = 1;
        }

        /* Send the buffer off to the host.  Note this will return when complete */
        XUD_SetBuffer(ep_hid, (char *)p_reportBuffer, 4);
      }
    }
    // End
}
